/*
 * Copyright (c) 2007 Xilinx, Inc.  All rights reserved.
 *
 * Xilinx, Inc.
 * XILINX IS PROVIDING THIS DESIGN, CODE, OR INFORMATION "AS IS" AS A
 * COURTESY TO YOU.  BY PROVIDING THIS DESIGN, CODE, OR INFORMATION AS
 * ONE POSSIBLE      IMPLEMENTATION OF THIS FEATURE, APPLICATION OR
 * STANDARD, XILINX IS MAKING NO REPRESENTATION THAT THIS IMPLEMENTATION
 * IS FREE FROM ANY CLAIMS OF INFRINGEMENT, AND YOU ARE RESPONSIBLE
 * FOR OBTAINING ANY RIGHTS YOU MAY REQUIRE FOR YOUR IMPLEMENTATION.
 * XILINX EXPRESSLY DISCLAIMS ANY WARRANTY WHATSOEVER WITH RESPECT TO
 * THE ADEQUACY OF THE IMPLEMENTATION, INCLUDING BUT NOT LIMITED TO
 * ANY WARRANTIES OR REPRESENTATIONS THAT THIS IMPLEMENTATION IS FREE
 * FROM CLAIMS OF INFRINGEMENT, IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

/* webserver.c: An example Webserver application using the RAW API */

#include <stdio.h>
#include <string.h>

#include "lwip/err.h"
#include "lwip/tcp.h"
#include "xil_printf.h"
#include "webserver.h"
#include <stdlib.h>
#include "crypto_helpers.h"
#include "platform_gpio.h"
#include "platform_fs.h" // Needed for platform_init_fs()
#include "lwip/sys.h"
#include "lwip/tcp.h"
#include "lwip/timeouts.h"
static struct tcp_pcb *websocket_pcb = NULL;

// Prototypes for functions defined in other files
extern float get_digital_input_voltage(void); // From main.c
extern int generate_response(struct tcp_pcb *pcb, char *http_req, int http_req_len);
extern http_arg *palloc_arg();
extern void pfree_arg(http_arg *);

/* static variables controlling debug printf's in this file */
static int g_webserver_debug = 0;

static unsigned http_port = 80;
static unsigned http_server_running = 0;

// Function Prototypes (needed to avoid 'invalid storage class' if defined later)
err_t http_sent_callback(void *arg, struct tcp_pcb *tpcb, u16_t len);
err_t http_recv_callback(void *arg, struct tcp_pcb *tpcb, struct pbuf *p, err_t err);
static err_t http_accept_callback(void *arg, struct tcp_pcb *newpcb, err_t err);
int do_websocket_upgrade(struct tcp_pcb *pcb, char *req, int rlen);
void websocket_stream_data(void);
void setup_http_arg(struct tcp_pcb *pcb, int l, char *b);
void print_web_app_header();
int send_websocket_frame(struct tcp_pcb *pcb, const char *payload, int len);
int transfer_web_data() {
	return 0;
}

// --- CALLBACK FUNCTIONS ---

/**
 * Handles continuous sending of large in-memory files (like index.html) via chunking.
 */
err_t http_sent_callback(void *arg, struct tcp_pcb *tpcb, u16_t len)
{
	int sndbuf;
	http_arg *a = (http_arg*)arg;

	// 1. WebSocket Check (Do not close this)
	if (tpcb == websocket_pcb) {
	    return ERR_OK;
	}

    // 2. Handle closed/closing connections (and free resources)
	if (tpcb->state > ESTABLISHED) {
        if (a) {
            free(a);
            tcp_arg(tpcb, NULL);
        }
		tcp_close(tpcb);
        return ERR_OK;
	}

    // 3. Chunk Sending Logic for index.html (The ONLY data transfer logic)
	if (a) {
        // If there is still data left to send (count < len)
        if (a->count < a->len) {
            sndbuf = tcp_sndbuf(tpcb);
            int remaining = a->len - a->count;
            int to_write = (sndbuf < remaining) ? sndbuf : remaining;

            int is_last_chunk = (to_write == remaining);

            if (to_write > 0) {
                // Use the new union path for buffer access
                if (tcp_write(tpcb, a->data.memory.buf + a->count, to_write, TCP_WRITE_FLAG_COPY | (is_last_chunk ? 1 : TCP_WRITE_FLAG_MORE)) != ERR_OK) {
                    xil_printf("Chunk write failed. Closing.\r\n");
                    free(a);
                    tcp_arg(tpcb, NULL);
                    tcp_close(tpcb);
                    return ERR_ABRT;
                }
                a->count += to_write;
                tcp_output(tpcb);
            }
        }

        // 4. If transmission is complete, free resources and close
        if (a->count >= a->len) {
            free(a);
            tcp_arg(tpcb, NULL);
            tcp_close(tpcb);
        }
	}

	return ERR_OK;
}


/**
 * Called when data is received on an established connection.
 */
err_t http_recv_callback(void *arg, struct tcp_pcb *tpcb, struct pbuf *p, err_t err)
{
	http_arg *a = (http_arg*)arg;

	if (g_webserver_debug)
		xil_printf("%d (%d): R%d %d..\r\n", a?a->count:0, tpcb->state, p->len, p->tot_len);

	/* do not read the packet if we are not in ESTABLISHED state */
	if (tpcb->state >= 5 && tpcb->state <= 8) {
		if (a) {
			pfree_arg(a);
			a = NULL;
		}
		tcp_close(tpcb);
		return ERR_OK; // FIX: Return error code
	} else if (tpcb->state > 8) {
		return ERR_OK; // FIX: Return error code
	}

	// Handle WebSocket received data
	if (tpcb == websocket_pcb) {
	    xil_printf("WS: Client disconnected or sent data. Closing connection.\r\n");
	    websocket_pcb = NULL;
	    tcp_close(tpcb);
	    pbuf_free(p);
	    return ERR_OK;
	}

	/* acknowledge that we've read the payload */
	tcp_recved(tpcb, p->len);

	/* read and decipher the request */
	generate_response(tpcb, p->payload, p->len);

	/* free received packet */
	pbuf_free(p);

	return ERR_OK;
}


/**
 * Called when a new connection is accepted on the listening PCB.
 */
static err_t http_accept_callback(void *arg, struct tcp_pcb *newpcb, err_t err)
{
	/* keep a count of connection # */
	tcp_arg(newpcb, (void*)palloc_arg());

	tcp_recv(newpcb, http_recv_callback);
	tcp_sent(newpcb, http_sent_callback);

	return ERR_OK;
}

// --- SERVER INITIALIZATION ---

int start_web_application()
{
	struct tcp_pcb *pcb;
	err_t err;

	/* initialize file system layer */
	platform_init_fs();

	/* initialize devices */
	platform_init_gpios();

	/* create new TCP PCB structure */
	pcb = tcp_new();
	if (!pcb) {
		xil_printf("Error creating PCB. Out of Memory\r\n");
		return -1;
	}

	/* bind to http port 80 */
	err = tcp_bind(pcb, IP_ADDR_ANY, http_port);
	if (err != ERR_OK) {
		xil_printf("Unable to bind to port 80: err = %d\r\n", err);
		return -2;
	}

	/* we do not need any arguments to the first callback */
	tcp_arg(pcb, NULL);

	/* listen for connections */
	pcb = tcp_listen(pcb);
	if (!pcb) {
		xil_printf("Out of memory while tcp_listen\r\n");
		return -3;
	}

	/* specify callback to use for incoming connections */
	tcp_accept(pcb, http_accept_callback);

    http_server_running = 1;
	return 0;
}

// --- UTILITY FUNCTIONS ---

/**
 * Sets up the http_arg structure for in-memory file chunking.
 */
void setup_http_arg(struct tcp_pcb *pcb, int l, char *b) {
    // We are using malloc here directly, assuming palloc_arg is primarily for MFS
    http_arg *a = (http_arg *)malloc(sizeof(http_arg));
    if (a) {
        a->len = l;
        a->data.memory.buf = b; // Use new union path
        a->count = 0;

        // Ensure old MFS fields are NULL/0 if needed
        a->data.file.fd = 0;
        a->data.file.fsize = 0;

        tcp_arg(pcb, a);
        tcp_sent(pcb, http_sent_callback); // This registers the continuation function
    } else {
        xil_printf("Error: Failed to allocate http_arg.\r\n");
    }
}

void print_web_app_header()
{
    xil_printf("%20s %6d %s\r\n", "http server",
                         http_port,
                         "Point your web browser to http://192.168.1.25");
}

// --- WEBSOCKET FUNCTIONS ---

#define KEY_LEN 24
#define MAGIC_KEY "258EAFA5-E914-47DA-95CA-C5AB0DC85B11" // WebSocket GUID

/**
 * Handle the WebSocket connection upgrade handshake.
 */
int do_websocket_upgrade(struct tcp_pcb *pcb, char *req, int rlen) {
    char *key_start, client_key[KEY_LEN + 1];
    char key_concat[KEY_LEN + sizeof(MAGIC_KEY) + 1];
    char server_key[29]; // 28 chars for Base64 + 1 for null
    char response_header[256];

    // 1. Extract the Client Key
    key_start = strstr(req, "Sec-WebSocket-Key:");
    if (!key_start) { /* Send 400 Bad Request if missing */ return -1; }

    key_start += strlen("Sec-WebSocket-Key:");
    while (*key_start == ' ') key_start++;
    strncpy(client_key, key_start, KEY_LEN);
    client_key[KEY_LEN] = 0;

    // 2. Concatenate with Magic Key
    strcpy(key_concat, client_key);
    strcat(key_concat, MAGIC_KEY);

    // 3. Compute the Accept Key (SHA1 + Base64)
    sha1_and_base64_encode(key_concat, server_key);

    // 4. Send the Handshake Response
    sprintf(response_header,
            "HTTP/1.1 101 Switching Protocols\r\n"
            "Upgrade: websocket\r\n"
            "Connection: Upgrade\r\n"
            "Sec-WebSocket-Accept: %s\r\n\r\n", server_key);

    if (tcp_write(pcb, response_header, strlen(response_header), 1) != ERR_OK) return -1;
    tcp_output(pcb);

    // 5. Store the PCB for Persistence
    if (websocket_pcb != NULL) tcp_close(websocket_pcb);
    websocket_pcb = pcb;
    tcp_arg(pcb, NULL); // Remove argument to stop default cleanup

    xil_printf("WebSocket Handshake Complete.\r\n");
    return 0;
}

/**
 * Helper function to send a simple WebSocket text frame.
 */
int send_websocket_frame(struct tcp_pcb *pcb, const char *payload, int len) {
    char header[10];
    int header_len = 2; // For payloads < 126 bytes

    // FIN=1, Opcode=0x1 (Text) -> 0x81
    header[0] = 0x81;
    header[1] = (char)len;

    if (tcp_sndbuf(pcb) < len + header_len) return -1;

    // Write header and payload
    if (tcp_write(pcb, header, header_len, 0) != ERR_OK) return -1;
    if (tcp_write(pcb, payload, len, 1) != ERR_OK) return -1; // PSH flag on last write

    return 0;
}

/**
 * Called from main loop (main.c) to stream the latest voltage data.
 */
void websocket_stream_data(void) {
    if (websocket_pcb == NULL || websocket_pcb->state != ESTABLISHED) {
        websocket_pcb = NULL;
        return;
    }

    char payload[32];
    float voltage = get_digital_input_voltage();

    // Format the voltage as a raw float string (e.g., "0.895")
    snprintf(payload, sizeof(payload), "%.3f", voltage);

    if (send_websocket_frame(websocket_pcb, payload, strlen(payload)) != 0) {
        tcp_close(websocket_pcb);
        websocket_pcb = NULL;
    }

    tcp_output(websocket_pcb); // Push the data immediately
}
