/*
 * main.c
 *
 * Initializes hardware (AXI GPIO + XADC) and runs LwIP Webserver.
 * Streams real-time voltage data from AXI GPIO to the web waveform.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "xparameters.h"
#include "platform.h"
#include "platform_config.h"
#include "xil_printf.h"
#include "xsysmon.h"
#include "xgpio.h"
#include "xgpiops.h"
#include "xstatus.h"

/* LwIP specific headers */
#include "lwip/init.h"
#include "lwip/inet.h"
#include "lwip/ip_addr.h"
#include "netif/xadapter.h"

/* GPIO Headers */
#include "platform_gpio.h"

/* --- Global Instances --- */
XGpio Gpio;           /* AXI GPIO (PL) */
XGpioPs Gpiops;       /* PS GPIO (MIO) */
XSysMon XADCInst;     /* XADC */
struct netif echo_netif;

/* --- Macros --- */
#define AXI_GPIO_DEVICE_ID   XPAR_AXI_GPIO_0_DEVICE_ID
#define AXI_GPIO_CHANNEL_OUT 1
#define AXI_GPIO_CHANNEL_IN  2
#define XADC_DEVICE_ID       XPAR_SYSMON_0_DEVICE_ID
#define XADC_STRING_MAX_LEN  256

#define WAVEFORM_SAMPLE_DELAY_US 5000 // 5ms delay (~200 samples/sec)
#define SLOW_PRINT_RATE          200  // 200 * 5ms = 1s refresh for UART + HTML

/* --- Calibration Constants --- */
#define GAIN   0.95f    // Slope calibration factor
#define OFFSET -0.05f   // Offset correction (+50mV to fix under-read)

/* --- Function Prototypes --- */
extern void start_web_application(void);
extern void transfer_web_data(void);
extern void print_web_app_header(void);
extern void websocket_stream_data(void);

/* --- Globals shared with http_response.c --- */
volatile float LatestDigitalInputVoltage = 0.0f;
char XadcSensorString[XADC_STRING_MAX_LEN] = "Data loading...";

/* --------------------------------------------------------------------
 * Utility: Print IP Info
 * -------------------------------------------------------------------- */
static void print_ip(char *msg, struct ip4_addr *ip)
{
    xil_printf("%s%d.%d.%d.%d\r\n", msg, ip4_addr1(ip), ip4_addr2(ip),
               ip4_addr3(ip), ip4_addr4(ip));
}

static void print_ip_settings(struct ip4_addr *ip, struct ip4_addr *mask, struct ip4_addr *gw)
{
    print_ip("Board IP: ", ip);
    print_ip("Netmask : ", mask);
    print_ip("Gateway : ", gw);
}

/* --------------------------------------------------------------------
 * Accessors for Webserver
 * -------------------------------------------------------------------- */
char* get_xadc_html_data() { return XadcSensorString; }
float get_digital_input_voltage() { return LatestDigitalInputVoltage; }

/* --------------------------------------------------------------------
 * MAIN FUNCTION
 * -------------------------------------------------------------------- */
int main()
{
    struct netif *netif = &echo_netif;
    struct ip4_addr ipaddr, netmask, gw;
    unsigned char mac_ethernet_address[] = {0x00, 0x0a, 0x35, 0x00, 0x01, 0x02};

    int Status;
    XSysMon_Config *ConfigPtr;

    xil_printf("\r\n--- Web Server Initialization ---\r\n");
    init_platform();

    xil_printf("Calibration: GAIN = %.3f, OFFSET = %.3fV\r\n", GAIN, OFFSET);

    /* ======================== AXI GPIO Init ======================== */
    Status = XGpio_Initialize(&Gpio, AXI_GPIO_DEVICE_ID);
    if (Status != XST_SUCCESS) {
        xil_printf("FATAL: AXI GPIO init failed!\r\n");
        return XST_FAILURE;
    }

    XGpio_SetDataDirection(&Gpio, AXI_GPIO_CHANNEL_OUT, 0x0);  // Output
    XGpio_SetDataDirection(&Gpio, AXI_GPIO_CHANNEL_IN,  0xFF); // Input

    xil_printf("AXI GPIO initialized. CH1=OUT, CH2=IN.\r\n");

    /* ======================== XADC Init ======================== */
    ConfigPtr = XSysMon_LookupConfig(XADC_DEVICE_ID);
    if (ConfigPtr == NULL) {
        xil_printf("XADC Config lookup failed!\r\n");
        return XST_FAILURE;
    }

    Status = XSysMon_CfgInitialize(&XADCInst, ConfigPtr, ConfigPtr->BaseAddress);
    if (Status != XST_SUCCESS) {
        xil_printf("XADC initialization failed!\r\n");
        return XST_FAILURE;
    }

    xil_printf("XADC initialized successfully.\r\n");

    /* ======================== LwIP Init ======================== */
    IP4_ADDR(&ipaddr, 192, 168, 1, 25);
    IP4_ADDR(&netmask, 255, 255, 255, 0);
    IP4_ADDR(&gw, 192, 168, 1, 1);
    print_ip_settings(&ipaddr, &netmask, &gw);

    lwip_init();

    if (!xemac_add(netif, &ipaddr, &netmask, &gw,
                   mac_ethernet_address, PLATFORM_EMAC_BASEADDR)) {
        xil_printf("Error adding network interface!\r\n");
        return -1;
    }

    netif_set_default(netif);
    platform_enable_interrupts();
    netif_set_up(netif);

    /* ======================== Start Web App ======================== */
    start_web_application();

    xil_printf("\r\n%-20s %-6s %s\r\n", "Server", "Port", "Connect With..");
    xil_printf("%-20s %-6s %s\r\n", "--------------------", "------", "--------------------");
    print_web_app_header();
    xil_printf("\r\n--- Initialization Complete ---\r\n");

    /* ======================== Main Loop ======================== */
    unsigned int slow_counter = 0;

    while (1) {
        xemacif_input(netif);
        transfer_web_data();

        /* --- Read Digital Voltage from AXI GPIO --- */
        u32 DigitalInputValue = XGpio_DiscreteRead(&Gpio, AXI_GPIO_CHANNEL_IN);

        /* Apply calibration (0–255 input  0–1 V range) */
        LatestDigitalInputVoltage = ((float)DigitalInputValue / 255.0f) * GAIN + OFFSET;
        if (LatestDigitalInputVoltage < 0.0f) LatestDigitalInputVoltage = 0.0f;
        if (LatestDigitalInputVoltage > 1.0f) LatestDigitalInputVoltage = 1.0f;

        /* Stream to waveform JSON (websocket) */
        websocket_stream_data();

        /* --- Slow debug and HTML update every ~1s --- */
        if (++slow_counter >= SLOW_PRINT_RATE) {
            u16 RawDataTemp = XSysMon_GetAdcData(&XADCInst, XSM_CH_TEMP);
            u16 RawDataVccInt = XSysMon_GetAdcData(&XADCInst, XSM_CH_VCCINT);
            u16 RawDataVccAux = XSysMon_GetAdcData(&XADCInst, XSM_CH_VCCAUX);

            float Temperature = XSysMon_RawToTemperature(RawDataTemp);
            float VccIntVoltage = XSysMon_RawToVoltage(RawDataVccInt);
            float VccAuxVoltage = XSysMon_RawToVoltage(RawDataVccAux);

            int TempInt = (int)Temperature;
            int TempDec = (int)((Temperature - TempInt) * 100);
            int VccIntInt = (int)VccIntVoltage;
            int VccIntDec = (int)((VccIntVoltage - VccIntInt) * 1000);
            int VccAuxInt = (int)VccAuxVoltage;
            int VccAuxDec = (int)((VccAuxVoltage - VccAuxInt) * 1000);
            int DigInt = (int)LatestDigitalInputVoltage;
            int DigDec = (int)((LatestDigitalInputVoltage - DigInt) * 1000);

            char DigitalStatus[9];
            for (int i = 0; i < 8; i++)
                DigitalStatus[i] = (DigitalInputValue & (1 << (7 - i))) ? '1' : '0';
            DigitalStatus[8] = '\0';

            xil_printf("Temp: %d.%02d°C | VCCINT: %d.%03dV | VCCAUX: %d.%03dV | GPIO Volt: %d.%03dV | GPIO Bits: %s\r\n",
                        TempInt, TempDec, VccIntInt, VccIntDec,
                        VccAuxInt, VccAuxDec, DigInt, DigDec, DigitalStatus);

            snprintf(XadcSensorString, XADC_STRING_MAX_LEN,
                     "<div class='sensor-data'>"
                     "  <p><strong>Temp:</strong> %d.%02d C</p>"
                     "  <p><strong>VCCINT:</strong> %d.%03d V</p>"
                     "  <p><strong>VCCAUX:</strong> %d.%03d V</p>"
                     "  <p><strong>Digital Input Voltage:</strong> <span class='text-red-400'>%d.%03d V</span></p>"
                     "</div>",
                     TempInt, TempDec, VccIntInt, VccIntDec,
                     VccAuxInt, VccAuxDec, DigInt, DigDec);

            slow_counter = 0;
        }

        usleep(WAVEFORM_SAMPLE_DELAY_US);
    }

    cleanup_platform();
    return 0;
}
