//http_response.c1

#include <string.h>
#include "mfs_config.h"
#include "lwip/inet.h"
#include "webserver.h"     // Assumed to contain generate_http_header() prototype
#include "platform_gpio.h" // For toggle_leds()
#include "xil_printf.h"
#include "lwip/tcp.h" // Add this if it's not present, needed for tcp_pcb
#include <stdlib.h>   // Required for malloc in setup_http_arg
#include "lwip/priv/tcp_priv.h" // You might need this include for PCB_GET_ARG, try without it first
#include <math.h> // Required for sin() in the simulation
// --- NEW STRUCTURE AND EXTERNS FOR CHUNKING ---

// Structure to track the progress of a large file transfer

static double current_voltage = 1.0;

static double current_time = 0.0;
static int voltage_counter = 0;
// Prototypes for functions defined in webserver.c
extern err_t http_sent_callback(void *arg, struct tcp_pcb *tpcb, u16_t len);
extern void setup_http_arg(struct tcp_pcb *pcb, int l, char *b);

extern char* get_xadc_html_data(void); // <-- Add this at the top

/* --- 404 Page (Unchanged) --- */
char *notfound_header =
    "<html><head><title>404</title>"
    "<style>div#request {background: #eeeeee}</style></head>"
    "<body><h1>404 Page Not Found</h1><div id=\"request\">";

char *notfound_footer =
    "</div></body></html>";

const char *index_html =
    "<html>"
    "<head>"
    "  <meta charset=\"UTF-8\">"
    "  <link rel=\"stylesheet\" href=\"https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css\">"
    "  <title>Numato Web Server</title>"
    "  <script src=\"https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js\"></script>"
    "  <style>"
    "    body { margin: 0; font-family: Arial, sans-serif; background-color: #f4f6f9; }"
    "    .login-container { height: 100vh; display: flex; justify-content: center; align-items: center; flex-direction: column; }"
    "    .login-box { background: white; padding: 60px; border-radius: 50px; box-shadow: 0 0 10px rgba(0,0,0,0.1); text-align: center; width: 100%; max-width: 260px; }"
    "    .login-box input { width: 100%; padding: 10px; margin: 10px 0; font-size: 16px; border: 1px solid #ccc; border-radius: 5px; }"
    "    .login-box button { padding: 10px 20px; background-color: #007BFF; color: white; font-size: 16px; border: none; border-radius: 5px; cursor: pointer; }"
    "    .login-box button:hover { background-color: #0056b3; }"
    "    .container { display: none; height: 100vh; }"
    "    .sidebar { width: 250px; background-color: #1f2937; color: white; display: flex; flex-direction: column; padding: 20px; }"
    "    .sidebar img { width: 200px; margin-bottom: 30px; }"
    "    .sidebar a { padding: 15px; margin: 5px 0; text-decoration: none; color: white; border-radius: 5px; transition: 0.3s; }"
    "    .sidebar a:hover, .sidebar a.active { background-color: #374151; }"
    "    .content { flex: 1; padding: 40px; overflow-y: auto; }"
    "    .card { background-color: white; border-radius: 10px; padding: 30px; box-shadow: 0 4px 8px rgba(0,0,0,0.1); }"
    "    .button { padding: 10px 20px; margin: 10px; border: 2px solid #007BFF; border-radius: 5px; background-color: white; color: #007BFF; cursor: pointer; font-size: 16px; transition: 0.3s; }"
    "    .button:hover { background-color: #007BFF; color: white; }"
    "    .button.active { background-color: #007BFF; color: white; }"
    "    .label { font-weight: bold; font-size: 18px; }"
    "  </style>"
    "  <script>"
    "    function showSection(id) {"
    "      document.getElementById('home').style.display = 'none';"
    "      document.getElementById('demo').style.display = 'none';"
    "      document.getElementById('support').style.display = 'none';"
    "      document.getElementById(id).style.display = 'block';"
    "      var links = document.querySelectorAll('.sidebar a');"
    "      links.forEach(link => link.classList.remove('active'));"
    "      document.getElementById(id + '-link').classList.add('active');"
    "    }"
    "    function toggleLED() {"
    "      const button = document.getElementById('led-button');"
    "      const statusSpan = document.getElementById('gpio-status');"
    "      fetch('/', {"
    "        method: 'POST',"
    "        headers: {'Content-Type': 'application/x-www-form-urlencoded'},"
    "        body: 'action=Toggle'"
    "      })"
    "      .then(response => response.text())"
    "      .then(status => {"
    "        if (status.trim() === '1') {"
    "          button.classList.add('active');"
    "          statusSpan.innerText = 'ON';"
    "          statusSpan.style.color = '#388E3C';"
    "        } else if (status.trim() === '0') {"
    "          button.classList.remove('active');"
    "          statusSpan.innerText = 'OFF';"
    "          statusSpan.style.color = '#D32F2F';"
    "        }"
    "      })"
    "      .catch(error => {"
    "        console.error('Fetch error:', error);"
    "        alert('Communication error with board.');"
    "      });"
    "    }"
    "    function updateXadcValues() {"
    "      fetch('/xadc')"
    "        .then(response => response.text())"
    "        .then(html => {"
    "          document.getElementById('xadc-values').innerHTML = html;"
    "        })"
    "        .catch(error => {"
    "          document.getElementById('xadc-values').innerText = 'Sensor error!';"
    "        });"
    "    }"
    "    let liveChart;"
		"function setupWaveform() {"
		"  const ctx = document.getElementById('voltageChart').getContext('2d');"
		"  liveChart = new Chart(ctx, {"
		"    type: 'line',"
		"    data: {"
		"      labels: [],"
		"      datasets: [{"
		"        label: 'Digital Input Voltage (V)',"
		"        borderColor: 'rgb(0, 119, 204)',"
		"        borderWidth: 2,"
		"        data: [],"
		"        tension: 0.4,"
		"        pointRadius: 0"
		"      }]"
		"    },"
		"    options: {"
		"      responsive: true,"
		"      maintainAspectRatio: false,"
		"      animation: { duration: 200, easing: 'linear' },"
		"      scales: {"
		"        x: { title: { display: true, text: 'Time (s)' } },"
		"        y: { title: { display: true, text: 'Voltage (V)' }, min: 0, max: 1.01 ,ticks: { stepSize: 0.1} }"
		"      }"
		"    }"
		"  });"
		"}"

    "    function updateWaveform() {"
    "      fetch('/waveform')"
    "        .then(response => response.json())"
    "        .then(data => {"
    "          liveChart.data.labels.push(data.time);"
    "          liveChart.data.datasets[0].data.push(data.voltage);"
    "          const maxDataPoints = 20;"
    "          if (liveChart.data.labels.length > maxDataPoints) {"
    "            liveChart.data.labels.shift();"
    "            liveChart.data.datasets[0].data.shift();"
    "          }"
    "          liveChart.update('none');"
    "        })"
    "        .catch(error => { console.error('Waveform fetch error:', error); });"
    "    }"
    "    window.onload = function() { document.getElementById('username').focus(); };"
    "    function login(event) {"
    "      event.preventDefault();"
    "      var user = document.getElementById('username').value;"
    "      var pass = document.getElementById('password').value;"
    "      if (user === 'admin' && pass === 'admin') {"
    "        document.getElementById('login-page').style.display = 'none';"
    "        document.getElementById('main-page').style.display = 'flex';"
    "        showSection('home');"
    "        setInterval(updateXadcValues, 1000);"
    "        setupWaveform();"
    "        setInterval(updateWaveform, 200);"
    "      } else {"
    "        alert('Invalid credentials!');"
    "      }"
    "    }"
    "  </script>"
    "</head>"
    "<body>"
    "  <div id='login-page' class='login-container'>"
    "    <img src='https://numato.com/wp-content/uploads/2022/01/Numato_Logo.png' style='width: 300px; margin-bottom: 30px;'>"
    "    <form class='login-box' onsubmit='login(event)'>"
    "      <h2>Login</h2>"
    "      <input type='text' id='username' placeholder='Username' required>"
    "      <input type='password' id='password' placeholder='Password' required>"
    "      <button type='submit'>Login</button>"
    "    </form>"
    "  </div>"
    "  <div id='main-page' class='container'>"
    "    <div class='sidebar'>"
    "      <img src='https://numato.com/wp-content/uploads/2022/01/Numato_Logo.png'>"
    "      <a href='#' id='home-link' onclick='showSection(\"home\")'>Home</a>"
    "      <a href='#' id='demo-link' onclick='showSection(\"demo\")'>Demo</a>"
    "      <a href='#' id='support-link' onclick='showSection(\"support\")'>Support</a>"
    "    </div>"
    "    <div class='content'>"
    "      <div id='home' class='card' style='display:none;'>"
    "        <h2 style='text-align:center;'>TityraCore Zynq 7000 FPGA Development Board</h2>"
    "        <div style='text-align: center;'>"
    "          <img src='https://numato.com/wp-content/uploads/2025/05/NLSOMMZ701_001.png' style='width:100%; max-width:500px;'>"
    "        </div>"
    "        <h3 style='margin-top: 20px;'>Board Features</h3>"
    "        <style>"
    "          .features-list { font-family: Arial, sans-serif; font-size: 14px; text-align: left; margin-left: 0; padding-left: 0; line-height: 1.4; }"
    "          .features-list p::before { content: '> '; color: black; margin-right: 5px; }"
    "        </style>"
    "        <div class='features-list'>"
    "          <p><strong>Device:</strong> XC7Z020 in CLG484 package, Speed Grade: -1.</p>"
    "          <p><strong>Memory:</strong> 128 Mb Quad bit SPI flash, microSD slot, and 8 GB eMMC.</p>"
    "          <p><strong>Interfaces:</strong> Gigabit Ethernet, HDMI, MIPI, and High-Speed USB 2.0 OTG.</p>"
    "        </div>"
    "        <h3 style='margin-top: 20px;'>Applications</h3>"
    "        <div class='features-list'>"
    "          <p>Product Prototype Development.</p>"
    "          <p>Accelerated computing integration.</p>"
    "          <p>Custom embedded processor testing.</p>"
    "          <p>Signal Processing.</p>"
    "          <p>Communication device development.</p>"
    "          <p>Educational tool for Schools and Universities.</p>"
    "          <p>Video processing.</p>"
    "        </div>"
    "      </div>"
    "      <div id='demo' class='card' style='display:none;'>"
    "        <div style='text-align: center;'>"
    "          <img src='https://numato.com/help/wp-content/uploads/2024/01/Tityra_SODIMM_wiring_diagram.png' style='width:100%; max-width:700px;'>"
    "        </div>"
    "        <div class='card' style='width: 90%; margin: 50px auto; padding: 20px;'>"
    "          <h2>Live Digital Input Waveform</h2>"
    "          <div style='height: 350px;'>"
    "            <canvas id='voltageChart'></canvas>"
    "          </div>"
    "        </div>"
    "        <div class='card' style='width: 400px; margin: 50px auto; padding: 20px; text-align: center;'>"
    "          <h2>Web Server Demo</h2>"
    "          <p>Control the board peripherals from here.</p>"
    "          <div>"
    "            <span class='label'>Toggle SCL LED:</span>"
    "            <button type='button' id='led-button' class='button' onclick='toggleLED();'>Toggle</button>"
    "            <p style='font-size: 18px; margin-top: 15px;'>Status: <span id='gpio-status' style='font-weight: bold; color: #D32F2F;'>OFF</span></p>"
    "          </div>"
    "          <div style='margin-top:25px;'>"
    "            <span class='label'>Live XADC Sensor Readings:</span>"
    "            <div id='xadc-values' style='padding:10px; background:#efefef; margin-top:8px;'>"
    "              Loading XADC..."
    "            </div>"
    "          </div>"
    "        </div>"
    "      </div>"
    "      <div id='support' class='card' style='display:none;'>"
    "        <h2>Support</h2>"
    "        <p>Visit the official board page for more info:</p>"
    "        <p><a href='https://numato.com/product/tityra-core-zynq-7000-soc-som/' target='_blank'>TityraCore Zynq 7000 SoC SoM</a></p>"
    "      </div>"
    "    </div>"
    "  </div>"
    "</body>"
    "</html>";

extern volatile float LatestDigitalInputVoltage;  // Already exists
static float last_valid_voltage = 0.0f;

char *get_waveform_json() {
    // 1. Static Buffer: Allocated once, lasts forever. This eliminates the
    // high-frequency malloc/free calls that cause heap fragmentation and crash.
    static char buf[128];
    static double current_time = 0.0; // Keeps track of time across calls

    // Increment time by 0.2s to match JS poll interval
    current_time += 0.2;
    if (current_time > 1000.0)
        current_time = 0.0;

    // Read the latest voltage directly from the global variable
    extern volatile float LatestDigitalInputVoltage;
    float voltage_now = LatestDigitalInputVoltage;

    // Safety clamp (Using 1.01V based on typical ADC range, overriding the 3.3V mistake)
    if (voltage_now < 0.0f)
        voltage_now = 0.0f;
    if (voltage_now > 1.01f) // Clamped for safety
        voltage_now = 1.01f;

    // 2. Complete snprintf logic
    // Format the JSON data into the static buffer. snprintf ensures no overflow.
    snprintf(buf, sizeof(buf), "{\"time\":%.2f, \"voltage\":%.3f}", current_time, voltage_now);

    // CRITICAL: Since memory was NOT allocated with malloc,
    // the caller (do_http_get) MUST NOT call free() on the returned pointer.
    return buf;
}


// --- GENERIC HTTP HEADER GENERATION (Example/Assumption) ---

// You must ensure this function correctly handles the "json" type.
// You likely already have this, but I include a simplified version for context.
// --- GENERIC HTTP HEADER GENERATION (Example/Assumption) ---
int generate_http_header(char *buf, const char *type, int len) {
    char content_type[64];

    if (strcmp(type, "json") == 0) {
        strcpy(content_type, "application/json");
    } else if (strcmp(type, "html") == 0) {
        strcpy(content_type, "text/html");
    } else if (strcmp(type, "txt") == 0) {
        strcpy(content_type, "text/plain");
    } else {
        strcpy(content_type, "application/octet-stream");
    }

    // Generate the standard HTTP 200 OK header
    return sprintf(buf, "HTTP/1.1 200 OK\r\n"
                        "Content-Type: %s\r\n"
                        "Content-Length: %d\r\n"
                        "Connection: close\r\n"
                        "Access-Control-Allow-Origin: *\r\n" // Optional, but good for embedded systems
                        "\r\n", content_type, len);
}


// --- CORRECTED: do_404 (Adding tcp_close for resource cleanup) ---
/* Function to handle 404 errors */
int do_404(struct tcp_pcb *pcb, char *req, int rlen) {
    // Note: notfound_header and notfound_footer are assumed to be defined globally
    int len, hlen;
    char buf[1024];
    err_t err;

    len = strlen(notfound_header) + strlen(notfound_footer) + rlen;
    // hlen is the size of the HTTP header being generated into 'buf'
    hlen = generate_http_header(buf, "html", len);

    if (tcp_sndbuf(pcb) < (hlen + len)) {
        xil_printf("cannot send 404 message, tcp_sndbuf = %d bytes, message length = %d bytes\r\n",
                    tcp_sndbuf(pcb), hlen + len);
        tcp_close(pcb); // Added close on error
        return -1;
    }

    // 1. Write Header: Use TCP_WRITE_FLAG_MORE
    if ((err = tcp_write(pcb, buf, hlen, TCP_WRITE_FLAG_MORE)) != ERR_OK) {
        xil_printf("%s: error (%d) writing 404 http header\r\n", __FUNCTION__, err);
        tcp_close(pcb); // Added close on error
        return -1;
    }

    // 2. Write HTML Body in parts. Use MORE flag until the last chunk.
    tcp_write(pcb, notfound_header, strlen(notfound_header), TCP_WRITE_FLAG_MORE);
    tcp_write(pcb, req, rlen, TCP_WRITE_FLAG_MORE);

    // 3. Write Footer: Use PSH flag (1) on the final chunk.
    tcp_write(pcb, notfound_footer, strlen(notfound_footer), 1);

    tcp_output(pcb); // Flush all written data

    //  FIX: Close the connection after sending the response to prevent PCB leaks
    tcp_close(pcb);

    return 0;
}

// --- CORRECTED: do_http_post (Adding tcp_close for resource cleanup) ---
/* Function to handle HTTP POST requests */
int do_http_post(struct tcp_pcb *pcb, char *req, int rlen) {
    // Note: toggle_leds is assumed to be defined externally
    int BUFSIZE = 1024;
    int n = 0;
    err_t err;

    char *body = strstr(req, "\r\n\r\n");
    if (body) {
        body += 4;

        if (strstr(body, "action=Toggle") != NULL) {
            n = toggle_leds();
            xil_printf("http POST: Pmod Pin State: %d\r\n", n);
        }

    } else {
        return do_404(pcb, req, rlen);
    }

    int len;
    char buf[BUFSIZE];
    char *p;

    // The length of the response is just '1' (for '0' or '1')
    len = generate_http_header(buf, "txt", 1);

    p = buf + len;
    *p++ = n ? '1' : '0';
    *p = 0;
    len++; // total length of header + '1' or '0'

    if (tcp_sndbuf(pcb) < len) {
        xil_printf("cannot send POST response, tcp_sndbuf = %d bytes\r\n", tcp_sndbuf(pcb));
        tcp_close(pcb); // Added close on error
        return -1;
    }

    // Write Header and Body (just '0' or '1') in one go. Use PSH flag (1).
    if ((err = tcp_write(pcb, buf, len, 1)) != ERR_OK) {
        xil_printf("%s: error (%d) writing POST response\r\n", __FUNCTION__, err);
        // Clean up on write error
        tcp_close(pcb);
        return -1;
    }

    tcp_output(pcb);

    //  FIX: Close the connection after sending the response to prevent PCB leaks
    tcp_close(pcb);

    return 0;
}

// --- CORRECTED: do_http_get (Adding tcp_close and removing free(data)) ---
/* Function to handle GET requests */
int do_http_get(struct tcp_pcb *pcb, char *req, int rlen) {
    // Note: get_waveform_json, get_xadc_html_data, do_websocket_upgrade, index_html, setup_http_arg, http_arg, and http_sent_callback are assumed to be defined elsewhere.

    // 1. WebSocket Upgrade Check
    if (!strncmp(req, "GET /ws", 7)) {
        return do_websocket_upgrade(pcb, req, rlen);
    }

    // 2. XADC Endpoint Check
    if (!strncmp(req, "GET /xadc", 9)) {
        char buf[1024];
        char *data = get_xadc_html_data(); // Assumed to return a buffer that needs freeing
        int len = strlen(data);
        int hlen = generate_http_header(buf, "html", len);

        // 1. Write Header: Use MORE flag
        if (tcp_write(pcb, buf, hlen, TCP_WRITE_FLAG_MORE) != ERR_OK) {
            xil_printf("error writing xadc http header\r\n");
            free(data); // Free buffer on error (assuming get_xadc_html_data uses malloc)
            tcp_close(pcb); // Added close on error
            return -1;
        }
        // 2. Write Data: Use PSH flag (1)
        if (tcp_write(pcb, data, len, 1) != ERR_OK) {
            xil_printf("error writing xadc data to socket\r\n");
            free(data); // Free buffer on error
            tcp_close(pcb); // Added close on error
            return -1;
        }

        free(data); // Free the buffer on success
        tcp_output(pcb);
        //  FIX: Close the connection after sending the response
        tcp_close(pcb);
        return 0;
    }

    // 3. NEW: WAVEFORM JSON ENDPOINT CHECK
    if (!strncmp(req, "GET /waveform", 13)) {
        char buf[1024];
        char *data = get_waveform_json(); // Returns STATIC buffer (NO MALLOC/FREE)
        int len = strlen(data);

        // CRITICAL: Use "json" type and the length of the JSON string
        int hlen = generate_http_header(buf, "json", len);

        xil_printf("http GET: serving /waveform data: %s\r\n", data);

        // 1. Write Header: Use MORE flag
        if (tcp_write(pcb, buf, hlen, TCP_WRITE_FLAG_MORE) != ERR_OK) {
            xil_printf("error writing waveform http header\r\n");
            //  FIX: REMOVED free(data) - data is STATIC
            tcp_close(pcb);
            return -1;
        }

        // 2. Write Data: Use PSH flag (1)
        if (tcp_write(pcb, data, len, 1) != ERR_OK) {
            xil_printf("error writing waveform data to socket\r\n");
            //  FIX: REMOVED free(data) - data is STATIC
            tcp_close(pcb);
            return -1;
        }

        // 3. Output and Cleanup
        //  FIX: REMOVED free(data) - data is STATIC
        tcp_output(pcb);
        //  FIX: Close the connection after sending the response to prevent PCB leaks
        tcp_close(pcb);
        return 0;
    }
    // END NEW: WAVEFORM JSON ENDPOINT CHECK

    // --- 4. Normal index.html handling (Requires Chunking) ---

    xil_printf("http GET: serving index.html\r\n");

    int total_len = strlen(index_html);
    char header_buf[2048];
    int hlen = generate_http_header(header_buf, "html", total_len);
    err_t err;
    http_arg *a = NULL;

    // Check if the total size (Header + Data) fits in the current LwIP buffer
    if ((hlen + total_len) <= tcp_sndbuf(pcb)) {
        // --- CASE 1: Full content fits (Unlikely, but safe) ---

        // 1. Send Header (Use MORE)
        if ((err = tcp_write(pcb, header_buf, hlen, TCP_WRITE_FLAG_MORE)) != ERR_OK) goto write_error;

        // 2. Send Full Data (Use PSH flag, 1)
        if ((err = tcp_write(pcb, (char *)index_html, total_len, 1)) != ERR_OK) goto write_error;

        // No chunking needed. Flush and close.
        tcp_output(pcb);
        tcp_close(pcb);
        return 0;
    }

    // --- CASE 2: Chunking required (The expected path for your large HTML) ---

    // 1. Set up the argument structure for the continuation
    setup_http_arg(pcb, total_len, (char *)index_html);
    // CRITICAL: Retrieve the pointer set by setup_http_arg
    a = (http_arg*)pcb->callback_arg;
    if (!a) {
        xil_printf("FATAL: Failed to retrieve allocated http_arg.\r\n");
        return -1;
    }

    // 2. Write HTTP Header (Must use MORE)
    if ((err = tcp_write(pcb, header_buf, hlen, TCP_WRITE_FLAG_MORE)) != ERR_OK) goto write_error;

    // 3. Calculate how much data can fit in the *remaining* buffer
    int data_to_write = tcp_sndbuf(pcb) - hlen;

    if (data_to_write < 0) {
        xil_printf("CRITICAL: Header size (%d) exceeds TCP_SND_BUF (%d)!\r\n", hlen, tcp_sndbuf(pcb));
        data_to_write = 0;
    }

    if (data_to_write > total_len) data_to_write = total_len;

    // 4. Write the first data chunk
    if (data_to_write > 0) {
        // Use TCP_WRITE_FLAG_COPY and TCP_WRITE_FLAG_MORE
        if ((err = tcp_write(pcb, a->data.memory.buf, data_to_write, TCP_WRITE_FLAG_COPY | TCP_WRITE_FLAG_MORE)) != ERR_OK) goto write_error;
        a->count = data_to_write; // Mark the first chunk as sent
    } else {
        // Data to write is 0, first chunk will be sent by the callback
        a->count = 0;
    }

    // 5. Install the callback to send the remaining chunks
    tcp_sent(pcb, http_sent_callback);

    tcp_output(pcb); // Flush the initial header and chunk
    return 0;

write_error:
    xil_printf("error writing index.html header/chunk (err=%d)\r\n", err);
    // Free argument and close connection on error
    if (a) free(a);
    tcp_arg(pcb, NULL);
    tcp_close(pcb);
    return -1;
}

/* Function to decode the HTTP request type */
enum http_req_type { HTTP_GET, HTTP_POST, HTTP_UNKNOWN };
enum http_req_type decode_http_request(char *req, int l) {
    char *get_str = "GET";
    char *post_str = "POST";

    if (!strncmp(req, get_str, strlen(get_str)))
        return HTTP_GET;

    if (!strncmp(req, post_str, strlen(post_str)))
        return HTTP_POST;

    return HTTP_UNKNOWN;
}

/* Generate appropriate response based on the HTTP request */
int generate_response(struct tcp_pcb *pcb, char *http_req, int http_req_len) {
    enum http_req_type request_type = decode_http_request(http_req, http_req_len);

    switch(request_type) {
    case HTTP_GET:
        return do_http_get(pcb, http_req, http_req_len);
    case HTTP_POST:
        return do_http_post(pcb, http_req, http_req_len);
    default:
        xil_printf("request_type != GET|POST\r\n");
        return do_404(pcb, http_req, http_req_len);
    }
}
