--------------------------------------------------------------------------------
-- Author  : Rohit Singh
-- Module  : ft2232h_phy - rtl
-- Project : FT2232H Async FIFO

-- License : The MIT License
-- Copyright � 2015, 2016 Rohit K. Singh
--------------------------------------------------------------------------------

library IEEE;

use IEEE.STD_LOGIC_1164.all;
use IEEE.NUMERIC_STD.all;

entity ft2232h_phy is
  port(
    clk : in std_logic;
    rst : in std_logic;

    -- read channel
    rready : in  std_logic;
    rvalid : out std_logic;
    rdata  : out std_logic_vector(7 downto 0);

    -- write channel
    wready : out std_logic;
    wvalid : in  std_logic;
    wdata  : in  std_logic_vector(7 downto 0);

    -- FTDI FIFO external interface
    data  : inout std_logic_vector(7 downto 0);
    rxf_n : in    std_logic;
    txe_n : in    std_logic;
    rd_n  : out   std_logic;
    wr_n  : out   std_logic
    );
end ft2232h_phy;

architecture rtl of ft2232h_phy is
  signal data_o  : unsigned(7 downto 0) := (others => '0');
  signal data_i  : unsigned(7 downto 0) := (others => '0');
  signal data_en : std_logic            := '0';

  signal rxf_n_meta : std_logic := '1';
  signal txe_n_meta : std_logic := '1';
  signal rxf_n_sync : std_logic := '1';
  signal txe_n_sync : std_logic := '1';

  signal rd_n_q : std_logic := '1';
  signal wr_n_q : std_logic := '1';

  signal wready_q : std_logic := '0';
  signal rvalid_q : std_logic := '0';

  signal rd_reg : unsigned(7 downto 0) := (others => '0');
  signal wr_reg : unsigned(7 downto 0) := (others => '0');

  type states is (IDLE, RD_WAIT, RD_POST_WAIT, RD_DONE, WR_WAIT, WR_DONE);
  signal state : states := IDLE;

  type op is (OP_READ, OP_WRITE);
  signal prev_op : op := OP_WRITE;

  signal pending_read  : std_logic            := '0';
  signal pending_write : std_logic            := '0';
  signal counter       : unsigned(1 downto 0) := "00";


begin

  -- Bi-directional logic
  data   <= std_logic_vector(data_o) when data_en = '1' else (others => 'Z');
  data_i <= unsigned(data);

  rd_n <= rd_n_q;
  wr_n <= wr_n_q;

  -- Avoid metastability. Synchronize async signals  RXF and TXE
  synchronizer : process(clk)
  begin
    if rising_edge(clk) then
      rxf_n_meta <= rxf_n;
      txe_n_meta <= txe_n;
      rxf_n_sync <= rxf_n_meta;
      txe_n_sync <= txe_n_meta;
    end if;
  end process;

  rvalid   <= rvalid_q;
  wready   <= wready_q;
  wready_q <= (not txe_n_sync) and (not pending_write);

  main : process(clk, rst)
  begin
    if rst = '1' then
      counter  <= "00";
      rvalid_q <= '0';
      prev_op  <= OP_WRITE;
      state    <= IDLE;
      rd_n_q   <= '1';
      wr_n_q   <= '1';
      wr_reg   <= (others => '0');
      rd_reg   <= (others => '0');

    elsif rising_edge(clk) then

      if wvalid = '1' and wready_q = '1' then
        wr_reg        <= unsigned(wdata);
        pending_write <= '1';
      end if;

      if rxf_n_sync = '0' and rvalid_q = '0' and pending_read = '0' then
        pending_read <= '1';
      end if;

      if rvalid_q = '1' and rready = '1' then
        rvalid_q <= '0';
      end if;

      case state is

        when IDLE =>
          if (pending_read = '1' and pending_write = '1' and prev_op = OP_WRITE) or
            (pending_read = '1' and pending_write = '0') then

            rd_n_q  <= '0';
            state   <= RD_WAIT;
            prev_op <= OP_READ;

          elsif (pending_read = '1' and pending_write = '1' and prev_op = OP_READ) or
            (pending_read = '0' and pending_write = '1') then

            data_o  <= wr_reg;
            data_en <= '1';
            state   <= WR_WAIT;
            prev_op <= OP_WRITE;
          end if;

        when RD_WAIT =>
          counter <= counter + 1;
          if counter = "01" then
            state   <= RD_POST_WAIT;
            rd_reg  <= data_i;
            counter <= "00";
          end if;

        when RD_POST_WAIT =>
          counter <= counter + 1;
          if counter = "01" then
            rd_n_q  <= '1';
            state   <= RD_DONE;
            counter <= "00";
          end if;

        when RD_DONE =>
          if rxf_n_sync = '1' then
            state        <= IDLE;
            rvalid_q     <= '1';
            pending_read <= '0';
            rdata        <= std_logic_vector(rd_reg);
          end if;

        when WR_WAIT =>
          wr_n_q  <= '0';
          counter <= counter + 1;
          if counter = "11" then
            state   <= WR_DONE;
            counter <= "00";
            data_en <= '0';
          end if;

        when WR_DONE =>
          wr_n_q <= '1';
          if txe_n_sync = '1' then
            state         <= IDLE;
            pending_write <= '0';
          end if;

      end case;

    end if;

  end process;

end rtl;
