#include <stdio.h>
#include "xil_printf.h"
#include "xparameters.h"
#include "xstatus.h"
#include "xil_exception.h"
#include "xil_assert.h"
#include "xaxivdma.h"
#include "xaxivdma_i.h"
#include "xuartlite_l.h"

#define HRES 800
#define VRES 600
#define MEMORY_BASE XPAR_MIG7SERIES_0_BASEADDR
u32 srcBuffer = (MEMORY_BASE  + 0x1000000);

XAxiVdma InstancePtr;
XAxiVdma_DmaSetup ReadCfg;

const uint8_t HSVpower[121] =
{0, 2, 4, 6, 8, 11, 13, 15, 17, 19, 21, 23, 25, 28, 30, 32, 34, 36, 38, 40,
42, 45, 47, 49, 51, 53, 55, 57, 59, 62, 64, 66, 68, 70, 72, 74, 76, 79, 81,
83, 85, 87, 89, 91, 93, 96, 98, 100, 102, 104, 106, 108, 110, 113, 115, 117,
119, 121, 123, 125, 127, 130, 132, 134, 136, 138, 140, 142, 144, 147, 149,
151, 153, 155, 157, 159, 161, 164, 166, 168, 170, 172, 174, 176, 178, 181,
183, 185, 187, 189, 191, 193, 195, 198, 200, 202, 204, 206, 208, 210, 212,
215, 217, 219, 221, 223, 225, 227, 229, 232, 234, 236, 238, 240, 242, 244,
246, 249, 251, 253, 255};

void fill(u8 red, u8 green, u8 blue) {
	int x, y;
	u32 i = 0;

	u8 *src = (u8 *) srcBuffer;

	for (y=0; y < VRES; y++) {
		for (x=0; x < HRES; x++) {
			src[i++] = blue;
			src[i++] = green;
			src[i++] = red;
		}
	}
}


/* Cycle HSV colors
 * URL: http://www.instructables.com/id/How-to-Make-Proper-Rainbow-and-Random-Colors-With-/
 */
void powerHSV(int angle)
{
  u8 red, green, blue;
  if (angle<120) {red = HSVpower[120-angle]; green = HSVpower[angle]; blue = 0;} else
  if (angle<240) {red = 0;  green = HSVpower[240-angle]; blue = HSVpower[angle-120];} else
                 {red = HSVpower[angle-240]; green = 0; blue = HSVpower[360-angle];}
  fill(red, green, blue);
}


int run_frame_buffer(int device_id, int hsize, int vsize);

/* MANDELBROT implementation
 *   Copyright: Mike Field, mfield@concepts.co.nz
 *   URL: http://hamsterworks.co.nz/mediawiki/index.php/Mandelbrot
 */
unsigned char brot(float cx, float cy)
{
	int i = 0;
	float x = cx, y = cy;
	while(i < 255 && (x*x + y*y < 4.0))
	{
		float t = x;
		x = x*x - y*y + cx;
		y = 2*t*y + cy;
		i++;
	}
	return i;
}

int mandelbrot()
{
	u8 *src;
	int x, y;
	float scale = 4.0 / 2;
	float cx = 0.0, cy = 0.0;

	src = (u8 *) srcBuffer;

	u32 j = 0;
	for(y = 0; y < VRES; y++)
		for(x = 0; x < HRES; x++)
		{
			unsigned char i = brot(scale * (x-HRES/2) / (HRES/2.0) + cx, scale * (y-VRES/2) / (HRES/2.0) + cy);
			src[j++] = 255 * (i&3) / 3;
			src[j++] = 255 * ((i>>2) & 7) / 7;
			src[j++] = 255 * (i>>5) / 7;
		}

	return 0;
}

void cycle_colors() {
	for (int k = 0; k < 360; k += 36)
		powerHSV(k);
}

void uart_read_framebuffer(u32 hsize, u32 vsize) {
	const u8 magic[] = {'m', 'a', 'g', 'i', 'c'};
	u8 i = 0, ch;
	u8 found_magic = 1;

	for(i=0; i < sizeof(magic); i++) {
		ch = inbyte();
		if (ch != magic[i]) {
			found_magic = 0;
			break;
		}
	}

	u8 *src = (u8 *) srcBuffer;
	int x, y;

	u32 k = 0;
	if (!found_magic)
		return;
	else {
		for (y=0; y < VRES; y++) {
			for (x=0; x < HRES; x++) {
				src[k++] = inbyte(); // blue
				src[k++] = inbyte(); // green
				src[k++] = inbyte(); // red
			}
		}
	}
}

int main()
{
    int status;

    xil_printf("Numato Lab : Neso Framebuffer Example\r\n");
    xil_printf("_____________________________________\r\n\n");

    xil_printf("Framebuffer base address: 0x%X\r\n", srcBuffer);

    status = run_frame_buffer(0, HRES, VRES);
    if (status != XST_SUCCESS) {
		xil_printf("Failed to run framebuffer, status: 0x%X\r\n", status);
		return XST_FAILURE;
	}

    while(1) {
    	cycle_colors();
    	fill(0xff, 0xff, 0x00);
    	mandelbrot(0, 0, 4, srcBuffer);
    	fill(0xff, 0xff, 0xff);

    	/* Disabling the framebuffer load over UART capability.
    	 * If needed, then comment out the above 4 lines starting
    	 * with cycle_colors() and uncomment the while block below
    	 */
    	/*while (XUartLite_IsReceiveEmpty(STDIN_BASEADDRESS) != 1) {
    		uart_read_framebuffer(HRES, VRES);
    	}*/
    }

    return 0;
}

int run_frame_buffer(int device_id, int hsize, int vsize) {
	int status;
	XAxiVdma_Config *Config;

	Config = XAxiVdma_LookupConfig(device_id);
	if (!Config) {
		xil_printf("No video DMA found for ID %d\r\n", device_id );
		return XST_FAILURE;
	}

	/* Initialize DMA engine */
	status = XAxiVdma_CfgInitialize(&InstancePtr, Config, Config->BaseAddress);
	if (status != XST_SUCCESS) {
		xil_printf("Configuration Initialization failed, status: 0x%X\r\n", status);
		return status;
	}

	u32 stride = hsize * (Config->Mm2SStreamWidth>>3);

	/* ************************************************** */
	/*           Setup the read channel                   */
	/*                                                    */
	/* ************************************************** */
	ReadCfg.VertSizeInput       = vsize;
	ReadCfg.HoriSizeInput       = stride;
	ReadCfg.Stride              = stride;
	ReadCfg.FrameDelay          = 0;      /* This example does not test frame delay */
	ReadCfg.EnableCircularBuf   = 1;      /* Only 1 buffer, continuous loop */
	ReadCfg.EnableSync          = 0;      /* Gen-Lock */
	ReadCfg.PointNum            = 0;
	ReadCfg.EnableFrameCounter  = 0;      /* Endless transfers */
	ReadCfg.FixedFrameStoreAddr = 0;      /* We are not doing parking */

	status = XAxiVdma_DmaConfig(&InstancePtr, XAXIVDMA_READ, &ReadCfg);
	if (status != XST_SUCCESS) {
		xil_printf("Read channel config failed, status: 0x%X\r\n", status);
		return status;
	}

	// Initially populate framebuffer with complete white
	fill(0xff, 0xff, 0xff);

	/* Set the buffer addresses for transfer in the DMA engine. This is address first pixel of the framebuffer */
	status = XAxiVdma_DmaSetBufferAddr(&InstancePtr, XAXIVDMA_READ, (UINTPTR *) &srcBuffer);
	if (status != XST_SUCCESS) {
		xil_printf("Read channel set buffer address failed, status: 0x%X\r\n", status);
		return status;
	}
	/************* Read channel setup done ************** */



	/* ************************************************** */
	/*  Start the DMA engine (read channel) to transfer   */
	/*                                                    */
	/* ************************************************** */

	/* Start the Read channel of DMA Engine */
	status = XAxiVdma_DmaStart(&InstancePtr, XAXIVDMA_READ);
	if (status != XST_SUCCESS) {
		xil_printf("Failed to start DMA engine (read channel), status: 0x%X\r\n", status);
		return status;
	}
	/* ************ DMA engine start done *************** */

	return XST_SUCCESS;
}



