import sys
import time
import numpy as np
import pyqtgraph as pg
from PyQt5.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QPushButton, QLabel, QMessageBox, QHBoxLayout
)
from PyQt5.QtCore import QTimer

XDMA_H2C = "/dev/xdma0_h2c_0"
XDMA_C2H = "/dev/xdma0_c2h_0"
BUFFER_SIZE = 1024 * 1024  # 1 MB
PLOT_HISTORY = 100         # Number of points to keep in plot

class PCIELoopbackPlotApp(QWidget):
    def __init__(self):
        super().__init__()
        self.init_ui()
        self.plot_data = []

    def init_ui(self):
        self.setWindowTitle("PCIe Gen3 x1 Loopback Test with Plot")
        self.setGeometry(300, 300, 600, 400)

        # Layouts
        main_layout = QVBoxLayout()
        info_layout = QHBoxLayout()

        # Widgets
        self.status_label = QLabel("Status: Ready")
        self.throughput_label = QLabel("Throughput: N/A")
        self.start_btn = QPushButton("Start Test")
        self.start_btn.clicked.connect(self.run_test)

        info_layout.addWidget(self.status_label)
        info_layout.addWidget(self.throughput_label)
        info_layout.addWidget(self.start_btn)

        # Plot widget
        self.plot_widget = pg.PlotWidget()
        self.plot_widget.setTitle("DMA Loopback Throughput (MB/s)")
        self.plot_widget.setYRange(0, 1500)
        self.plot_widget.setLabel('left', 'Throughput', units='MB/s')
        self.plot_widget.setLabel('bottom', 'Test Count')

        self.curve = self.plot_widget.plot(pen='g')

        # Assemble layout
        main_layout.addLayout(info_layout)
        main_layout.addWidget(self.plot_widget)
        self.setLayout(main_layout)

    def run_test(self):
        try:
            with open(XDMA_H2C, "wb") as h2c, open(XDMA_C2H, "rb") as c2h:
                tx_buf = np.arange(BUFFER_SIZE, dtype=np.uint8).tobytes()

                start = time.time()
                h2c.write(tx_buf)
                h2c.flush()

                rx_buf = c2h.read(BUFFER_SIZE)
                end = time.time()

                # Calculate throughput
                duration = end - start
                mbps = (2 * BUFFER_SIZE / (1024 * 1024)) / duration

                # Data check
                errors = sum(a != b for a, b in zip(tx_buf, rx_buf))

                # Update status
                if errors == 0:
                    self.status_label.setText("✅ Verified")
                else:
                    self.status_label.setText(f"❌ Errors: {errors}")

                self.throughput_label.setText(f"Throughput: {mbps:.2f} MB/s")

                # Update plot
                self.plot_data.append(mbps)
                if len(self.plot_data) > PLOT_HISTORY:
                    self.plot_data.pop(0)
                self.curve.setData(self.plot_data)

        except FileNotFoundError:
            QMessageBox.critical(self, "XDMA Error", "XDMA device files not found")
            self.status_label.setText("Device not found.")
        except Exception as e:
            QMessageBox.critical(self, "Error", str(e))
            self.status_label.setText("Test failed.")

if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = PCIELoopbackPlotApp()
    window.show()
    sys.exit(app.exec_())
