import sys
import os
import time
import numpy as np
from PyQt5.QtWidgets import (
    QApplication, QWidget, QVBoxLayout, QPushButton, QLabel, QMessageBox
)

XDMA_H2C = "/dev/xdma0_h2c_0"
XDMA_C2H = "/dev/xdma0_c2h_0"
BUFFER_SIZE = 1024 * 1024  # 1 MB

class PCIELoopbackApp(QWidget):
    def __init__(self):
        super().__init__()
        self.init_ui()

    def init_ui(self):
        self.setWindowTitle("PCIe Gen3 x1 Loopback Test")
        self.setGeometry(300, 300, 400, 200)

        layout = QVBoxLayout()

        self.status_label = QLabel("Status: Ready")
        self.throughput_label = QLabel("Throughput: N/A")

        self.start_btn = QPushButton("Start DMA Loopback Test")
        self.start_btn.clicked.connect(self.run_test)

        layout.addWidget(self.status_label)
        layout.addWidget(self.throughput_label)
        layout.addWidget(self.start_btn)

        self.setLayout(layout)

    def run_test(self):
        try:
            with open(XDMA_H2C, "wb") as h2c, open(XDMA_C2H, "rb") as c2h:
                tx_buf = np.arange(BUFFER_SIZE, dtype=np.uint8).tobytes()

                start = time.time()
                h2c.write(tx_buf)
                h2c.flush()

                rx_buf = c2h.read(BUFFER_SIZE)
                end = time.time()

                errors = sum(a != b for a, b in zip(tx_buf, rx_buf))
                duration = end - start
                mbps = (2 * BUFFER_SIZE / (1024 * 1024)) / duration

                self.throughput_label.setText(f"Throughput: {mbps:.2f} MB/s")

                if errors == 0:
                    self.status_label.setText("✅ Data Verified: No Errors")
                else:
                    self.status_label.setText(f"❌ Errors Detected: {errors}")
        except FileNotFoundError:
            QMessageBox.critical(self, "XDMA Device Error", "XDMA device files not found.")
            self.status_label.setText("Error: Device file not found.")
        except Exception as e:
            QMessageBox.critical(self, "Error", str(e))
            self.status_label.setText("Test failed.")

if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = PCIELoopbackApp()
    window.show()
    sys.exit(app.exec_())
